
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "CAENRFIDLib_Light.h"
#include "host.h"

static bool onError(CAENRFIDErrorCodes ec);
static void printHex(uint8_t* vect, uint16_t length, char* result);
static CAENRFIDErrorCodes SimpleInventoryExample(CAENRFIDReader reader,
                                                 char* source);
static CAENRFIDErrorCodes ContinuousInventoryExample(CAENRFIDReader reader,
                                                    char* source,
                                                    uint32_t ms_timelen);
                                                    
#define IFERROR(X) {  if( onError(X) == true ) return (-1); }


int main()
{
    CAENRFIDErrorCodes ec;
    CAENRFIDReader reader = { 
                             .connect = _connect,
                             .disconnect = _disconnect,
                             .tx = _tx,
                             .rx = _rx,
                             .clear_rx_data = _clear_rx_data,
                             .enable_irqs = _enable_irqs,
                             .disable_irqs = _disable_irqs
                            };
    RS232_params port_params = {
                            .com = "/dev/ttyACM0",
                            .baudrate = 115200, 
                            .dataBits = 8,
                            .stopBits = 1,
                            .parity = 0,
                            .flowControl = 0,
                            };
    char model[MAX_MODEL_LENGTH];
    char serial[MAX_SERIAL_LENGTH];
    char source[MAX_LOGICAL_SOURCE_NAME];
    const uint32_t rf_power = 200;
	char ch;

    strcpy(source, "Source_0");
    
    //connect to reader
    ec = CAENRFID_Connect(&reader, CAENRFID_RS232, &port_params);
    IFERROR(ec);
    printf("Connect : %d\n", ec);

    //read reader model and serial number
    ec = CAENRFID_GetReaderInfo(&reader, model, serial);
    IFERROR(ec);
    printf("Reader info : %d, %s, %s\n", ec, model, serial);

    //configure RF power level
    ec = CAENRFID_SetPower(&reader, rf_power);
    IFERROR(ec);
    printf("Set power to %d mW : %d\n", rf_power, ec);
    
    //Simple inventory Example
    printf("\n***Simple Inventory example***\n");
    ec = SimpleInventoryExample(reader, source);
    IFERROR(ec);
    printf("*** end - Simple Inventory ***\n");
    
    //Continuous inventory Example
    printf("\n***Continuous Inventory example***\n");
    ec = ContinuousInventoryExample(reader, source, 5000);
    IFERROR(ec);
    printf("*** end - Continuous Inventory ***\n");
    
    //disconnect from reader
    ec = CAENRFID_Disconnect(&reader);
    IFERROR(ec);
    printf("\nDisconnect : %d\n", ec);

	//wait user input before exiting..
	scanf("%c", &ch);

    return (0);
}

static bool onError(CAENRFIDErrorCodes ec)
{
    if(ec != CAENRFID_StatusOK)
    {
        printf(" ****** ERROR (%d) ****** \n", ec);
        return true;
    }
    return false;
}

static void printHex(uint8_t* vect, uint16_t length, char* result)
{
    int i;
    char* r = result;
    
    for(i = 0; i < length; i++)
    {
        sprintf(r, "%02X", vect[i]);
        r += 2;
    }
    *r = '\0';
}

static CAENRFIDErrorCodes SimpleInventoryExample(CAENRFIDReader reader,
                                                 char* source)
{
    CAENRFIDErrorCodes ec;
    const uint16_t flag = 0; //flag configured for Standard inventory
    CAENRFIDTagList *tags, *aux;
    uint16_t numTags;
    char epcStr[2*MAX_ID_LENGTH + 1];
    tags = NULL;
    numTags = 0;
    
    //run one inventory round
	ec = CAENRFID_InventoryTag(&reader, source, 0, 0, 0, NULL, 0, flag,
		                     &tags, &numTags);
    printf("Tags detected : %d\n", numTags);
    //Print epc code of detected tags
    while(tags != NULL)
    {
        printHex(tags->Tag.ID, tags->Tag.Length, epcStr);
        printf("  %s\n", epcStr);
        aux = tags;
        tags = tags->Next;
        free(aux);
    }   
    return ec;  
}

static CAENRFIDErrorCodes ContinuousInventoryExample(CAENRFIDReader reader,
                                                    char* source,
                                                    uint32_t ms_timelen)
{
    CAENRFIDErrorCodes ec;
    const uint16_t flag = CONTINUOS + FRAMED; //flag configured for Continuous inventory
    uint64_t ms_time_0, ms_time;
    CAENRFIDTag tag;
    bool has_tag, has_result_code, send_abort, abort_sent;
    char epcStr[2*MAX_ID_LENGTH + 1];
    
    //set no limit on number of continuous inventory cycles for
    //selected source
    if((ec = CAENRFID_SetSourceConfiguration(&reader, source,
                            CONFIG_READCYCLE, 0)) != CAENRFID_StatusOK)
    {
        return ec;
    }
    printf("Set readcycles to no limit : %d\n", ec);
    
    //start inventory
    if((ec = CAENRFID_InventoryTag(&reader, source, 0, 0, 0, NULL, 0, flag,
                            NULL, NULL)) != CAENRFID_StatusOK)
    {
        return ec;
    }
    ms_time_0 = get_ms_timestamp();
    printf("Start Continuous Inventory : %d at %llu msec\n", ec, ms_time_0);
    
    //get and parse packets from reader
    has_tag = false;
    has_result_code = false;
    send_abort = false;
    abort_sent = false;
    memset(&tag, 0, sizeof(tag));
    while(1)
    {
        ec = CAENRFID_GetFramedTag(&reader, &has_tag, &tag,
                            &has_result_code);
        if(has_result_code)
        {
            //Inventory ended, with return value ec
            printf("Inventory result code : %d\n", ec);
            break;
        }
        if(has_tag)
        {
            //A tag was detected and returned by reader
            has_tag = false;
            printHex(tag.ID, tag.Length, epcStr);
            printf("  %s\n", epcStr);
            memset(&tag, 0, sizeof(tag));
        }
        if(ec != CAENRFID_StatusOK)
        {
            //Unwanted condition
            ;
        }
        if(!abort_sent)
        {
            ms_time = get_ms_timestamp();
            if(ms_time - ms_time_0 > ms_timelen) send_abort = true;
            if(send_abort)
            {
                //Send abort command to reader
                //-then continue parsing packets until 
                // inventory result code is returned-
                ec = CAENRFID_InventoryAbort(&reader);              
                printf("Inventory abort : %d at %llu msec\n", ec, ms_time);
                printf(" elapsed %llu msec\n", ms_time - ms_time_0);
                if(ec != CAENRFID_StatusOK) break;
                abort_sent = true;
            }
        }
    }
    if(ec == CAENRFID_StatusOK)
    {
        //restore default number of inventory cycles for selected source
        ec = CAENRFID_SetSourceConfiguration(&reader, source,
                                    CONFIG_READCYCLE, 1);
        printf("Set readcycles to 1 : %d\n", ec);
    }
    return ec;  
}
